/* 
 Name:
	Create3DSphere
 Version:
	3.2 (15 August 2010)
 Author:
	Charles Bordenave
 Description:  
	This script evenly distributes the selected layers on a 3D sphere of controllable size.
 Usage:
	In After Effects CS4 or later, run the script
	Select some layers (footage, comp, solid or text layers)  
	Specify the sphere radius (unchecking Uniform allows you to create an ellipsoid)
	Click on Distribute to position the layers
	Use the 3D Null to control the sphere (position, scale, rotation, radius)
*/

	
// This class represents the main class of the script
function Create3DSphere()
{
	// Variable used to keep track of 'this' reference
	var create3DSphere = this;
	
	// Create an instance of the utils class to use its functions
	var utils = new Create3DSphereUtils();

	// Script infos
	this.scriptMinSupportVersion = "9.0";
	this.scriptName = "Create3DSphere.jsx";	
	this.scriptVersion = "3.2";
	this.scriptTitle = "Create 3D Sphere";
	this.scriptCopyright = "Copyright (c) 2010 Charles Bordenave";
	this.scriptHomepage = "http://www.nabscripts.com";
	this.scriptDescription = {en:"This script evenly distributes the selected layers on a 3D sphere of controllable size.\\r\\rFor easy control, a 3D Null positioned at the center of the sphere center is parent of the sphere layers. It also has a slider that controls the radius of the sphere.", 
							  fr:"Ce script répartit de façon équidistante les calques sélectionnés sur une sphère 3D de taille contrôlable.\\r\\rPour un meilleur contrôle, un Nul 3D positionné au centre de la sphère est parent des calques de la sphère. Il possède aussi un curseur pour contrôler le rayon de la sphère."};
	this.scriptAbout = {en:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription), 
						fr:this.scriptName + ", v" + this.scriptVersion + "\\r" + this.scriptCopyright + "\\r" + this.scriptHomepage + "\\r\\r" + utils.loc(this.scriptDescription)};		
	this.scriptUsage = {en:	"\u25BA In After Effects CS4 or later, run the script \\r" +
							"\u25BA Select some layers (footage, comp, solid or text layers) \\r" +  
							"\u25BA Specify the sphere radius (unchecking Uniform allows you to create an ellipsoid) \\r" +
							"\u25BA Click on Distribute to position the layers \\r" +
							"\u25BA Use the 3D Null to control the sphere (position, scale, rotation, radius)",
						fr:	"\u25BA Dans After Effects CS4 ou supérieur, exécuter le script \\r" +	
							"\u25BA Sélectionner des calques (média, comp, solide ou texte) \\r" +
							"\u25BA Spécifier le rayon de la sphère (décocher Uniforme permet de créer une ellipsoide) \\r" +
							"\u25BA Cliquer sur Distribuer pour positionner les calques \\r" +
							"\u25BA Utiliser le Nul 3D pour contrôler la sphère (position, échelle, rotation, rayon)"};

	// Errors
	this.requirementErr = {en:"This script requires After Effects CS4 or later.", fr:"Ce script nécessite After Effects CS4 ou supérieur."};	
	this.noCompErr = {en:"A comp must be active.", fr:"Une composition doit être active."};
	this.selLayersErr = {en:"Select at least one layer (maximum 1000).", fr:"Sélectionnez au moins un calque (maximum 1000)."};
	this.processErr = {en:"An error occured during the process. This may be cause by a wrong type of layer (a light for instance) or by an resources error.", fr:"Une erreur est survenue lors du process. Elle peut être due à un mauvais type de calque (une lumière par exemple) ou à un problème de ressources."};
	this.resErr = {en:"An error occured when trying to open and read a resource file.", fr:"Une erreur est survenue lors de l'ouverture d'un fichier de ressource."};
	
	// UI strings & default settings 
	this.aboutBtnName = "?";
	this.radiusStName = {en:"Radius:", fr:"Rayon:"};
	this.xradiusEtDflt = 400;
	this.yradiusEtDflt = 400;
	this.zradiusEtDflt = 400;
	this.uniformCbDflt = true; // sphere by default
	this.runBtnName = {en:"Distribute", fr:"Distribuer"};

	this.controllerLayerName = {en:"Controller", fr:"Contrôleur"};
	this.radiusEffectName = {en:"Radius", fr:"Rayon"};
	this.xradiusEffectName = {en:"X Radius", fr:"Rayon X"};
	this.yradiusEffectName = {en:"Y Radius", fr:"Rayon Y"};
	this.zradiusEffectName = {en:"Z Radius", fr:"Rayon Z"};
	this.uniformCbName = {en:"Uniform", fr:"Uniforme"};
	
	// Expressions
	this.spherePositionExpression = 
	"L = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\");\r" +
	"r = L.effect(\"" + utils.loc(this.radiusEffectName) + "\")(1);\r" +
	"r * value;";
	
	this.sphereOrientationExpression = 
	"lookAt(position, thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\").anchorPoint);";
	
	this.ellipsoidPositionExpression = 
	"L = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\");\r" +
	"a = L.effect(\"" + utils.loc(this.xradiusEffectName) + "\")(1);\r" +
	"b = L.effect(\"" + utils.loc(this.yradiusEffectName) + "\")(1);\r" +
	"c = L.effect(\"" + utils.loc(this.zradiusEffectName) + "\")(1);\r" +
	"[a * value[0], b * value[1], c * value[2]];";
	
	this.ellipsoidOrientationExpression =
	"L = thisComp.layer(\"" + utils.loc(this.controllerLayerName) + "\");\r" +
	"a = L.effect(\"" + utils.loc(this.xradiusEffectName) + "\")(1);\r" +
	"b = L.effect(\"" + utils.loc(this.yradiusEffectName) + "\")(1);\r" +
	"c = L.effect(\"" + utils.loc(this.zradiusEffectName) + "\")(1);\r" +
	"if (a == 0) a = 0.0001;\r" +
	"if (b == 0) b = 0.0001;\r" +
	"if (c == 0) c = 0.0001;\r" +
	"v = position;\r" +
	"n = 2 * [v[0] / Math.pow(a,2), v[1] / Math.pow(b,2), v[2] / Math.pow(c,2)];\r" +
	"lookAt(position, position + n);";
	
	// Internal data
	this.m_xradius = this.xradiusEtDflt;
	this.m_yradius = this.yradiusEtDflt;
	this.m_zradius = this.zradiusEtDflt;
	this.m_uniform = this.uniformCbDflt;
	this.m_pts;
				
	// Creates and displays the script interface
	this.buildUI = function (thisObj)
	{
		// dockable panel or palette
		var pal = (thisObj instanceof Panel) ? thisObj : new Window("palette", this.scriptTitle, undefined, {resizeable:false});

		// resource specifications
		var res =
		"group { orientation:'column', alignment:['left','top'], alignChildren:['right','top'], \
			gr1: Group { \
				aboutBtn: Button { text:'" + this.aboutBtnName + "', preferredSize:[25,20] } \
			}, \
			gr2: Group { \
				radiusSt: StaticText { text:'" + utils.loc(this.radiusStName) + "' }, \
				xradiusEt: EditText { text:'" + this.xradiusEtDflt + "', characters:4 }, \
				yradiusEt: EditText { text:'" + this.yradiusEtDflt + "', characters:4 }, \
				zradiusEt: EditText { text:'" + this.zradiusEtDflt + "', characters:4 } \
			}, \
			gr3: Group { \
				uniformCb: Checkbox { text:'" + utils.loc(this.uniformCbName) + "' } \
			}, \
			gr4: Group { orientation:'row', alignment:['fill','top'], \
				runBtn: Button { text:'" + utils.loc(this.runBtnName) + "', alignment:['fill','center'] } \
			} \
		}"; 
		pal.gr = pal.add(res);
		
		pal.layout.layout(true);
		var adj = ($.os.indexOf("Win") != -1) ? 0 : 3;
		pal.gr.gr3.uniformCb.size.width += pal.gr.gr3.uniformCb.windowBounds.left - pal.gr.gr2.xradiusEt.windowBounds.left + adj;
		
		pal.gr.gr3.uniformCb.value = this.m_uniform;
		pal.gr.gr2.yradiusEt.visible/*enabled*/ = 
		pal.gr.gr2.zradiusEt.visible/*enabled*/ = !this.m_uniform;
					
		// event callbacks
		pal.gr.gr1.aboutBtn.onClick = function () 
		{ 
			utils.createAboutDlg(create3DSphere.scriptAbout, create3DSphere.scriptUsage); 
		};

		pal.gr.gr2.xradiusEt.onChange = pal.gr.gr2.xradiusEt.onChanging = function () 
		{ 
			if (isNaN(this.text)) this.text = create3DSphere.xradiusEtDflt;
			this.text = parseFloat(this.text);
			
			if (create3DSphere.m_uniform)
			{
				pal.gr.gr2.yradiusEt.text = 
				pal.gr.gr2.zradiusEt.text = this.text;
				
				pal.gr.gr2.yradiusEt.notify("onChange");
				pal.gr.gr2.zradiusEt.notify("onChange");
			}
			create3DSphere.m_xradius = parseFloat(this.text); 
		};
		
		pal.gr.gr2.yradiusEt.onChange = function () 
		{ 
			if (isNaN(this.text)) this.text = create3DSphere.yradiusEtDflt;
			this.text = parseFloat(this.text);
			create3DSphere.m_yradius = parseFloat(this.text); 
		};
		
		pal.gr.gr2.zradiusEt.onChange = function () 
		{ 
			if (isNaN(this.text)) this.text = create3DSphere.zradiusEtDflt;
			this.text = parseFloat(this.text);
			create3DSphere.m_zradius = parseFloat(this.text); 
		};
		
		pal.gr.gr3.uniformCb.onClick = function () 
		{
			if (this.value)
			{
				pal.gr.gr2.yradiusEt.text = 
				pal.gr.gr2.zradiusEt.text = pal.gr.gr2.xradiusEt.text;
					
				pal.gr.gr2.yradiusEt.visible/*enabled*/ = 
				pal.gr.gr2.zradiusEt.visible/*enabled*/ = false;
			}
			else
			{
				pal.gr.gr2.yradiusEt.visible/*enabled*/ = 
				pal.gr.gr2.zradiusEt.visible/*enabled*/ = true;
			}					 
			create3DSphere.m_uniform = this.value; 
		};
		
		pal.gr.gr4.runBtn.onClick = function () 
		{ 
			create3DSphere.createSphere(); 
		};
		
		// show user interface
		if (pal instanceof Window)
		{
			pal.center();
			pal.show();
		}
		else
		{
			pal.layout.layout(true);
		}	   
	};

	// Determines whether the active item is a composition  
	this.checkActiveItem = function () 
	{
		return !(app.project.activeItem instanceof CompItem);
	}; 

	// Move a layer to a given position
	this.repositionLayer = function (layer, pos)
	{
		var err = false;
		
		if (layer instanceof CameraLayer || layer instanceof LightLayer) err = true;
		else
		{
			var comp = layer.containingComp;		
			try 
			{
				layer.threeDLayer = true;
				layer.position.numKeys ? layer.position.setValueAtTime(comp.time, pos) : layer.position.setValue(pos);
			}
			catch(e)
			{
				err = true;
			}
		}
		return err;	
	};

	// Creates a 3D Null and make it parent of the layers passed as arguments
	this.addController = function (layers)
	{
		var comp = layers[0].containingComp;
		var controllerLayer = comp.layers.addNull();
		controllerLayer.name = utils.loc(this.controllerLayerName);
		controllerLayer.threeDLayer = true;
		controllerLayer.enabled = false;
		
		if (this.m_uniform)
		{
			// add radius slider 
			var radiusEffect = controllerLayer.Effects.addProperty("ADBE Slider Control");
			radiusEffect.name = utils.loc(this.radiusEffectName);
			radiusEffect.property(1).setValue(this.m_xradius);   
		}
		else
		{
			// add slider for each radius
			var xradiusEffect = controllerLayer.Effects.addProperty("ADBE Slider Control");
			xradiusEffect.name = utils.loc(this.xradiusEffectName);
			xradiusEffect.property(1).setValue(this.m_xradius);
			
			var yradiusEffect = controllerLayer.Effects.addProperty("ADBE Slider Control");
			yradiusEffect.name = utils.loc(this.yradiusEffectName);
			yradiusEffect.property(1).setValue(this.m_yradius);
			
			var zradiusEffect = controllerLayer.Effects.addProperty("ADBE Slider Control");
			zradiusEffect.name = utils.loc(this.zradiusEffectName);
			zradiusEffect.property(1).setValue(this.m_zradius);			
		}
		
		controllerLayer.position.setValue([0,0,0]);
		
		var posExpr = this.m_uniform ? this.spherePositionExpression : this.ellipsoidPositionExpression;
		var oriExpr = this.m_uniform ? this.sphereOrientationExpression : this.ellipsoidOrientationExpression;
		
		// parent layers & add expressions
		for (var i = 0; i < layers.length; i++)  
		{
			layers[i].parent = controllerLayer;
			layers[i].position.expression = posExpr;
			layers[i].orientation.expression = oriExpr;
		}

		// reposition controller
		controllerLayer.position.setValue([comp.width / 2, comp.height / 2, 600]);
		
	};
			
	// Functional part of the script: distributes selected layers on 3D sphere			   
	this.createSphere = function ()
	{
		try
		{
			var comp = app.project.activeItem;
			var err = this.noCompErr;
			if (this.checkActiveItem(comp)) throw(err);
					
			var selLayers = comp.selectedLayers;
			var err = this.selLayersErr;
			if (selLayers.length < 1 || selLayers.length > 1000) throw(err);
			
			var resDir = $.fileName;
			resDir = resDir.substring(0,resDir.lastIndexOf("/") + 1);
			var resFileName = resDir + "(Create3DSphere_Resources)/n" + selLayers.length + ".jsxinc";  
			
			var err = this.resErr; 
			var file = new File(resFileName);
			if (!file.exists || !file.open("r")) throw(err);
			
			var content = file.read();
			file.close();
				
			this.m_pts = content.split(", ");
			this.m_pts[0] = this.m_pts[0].substring(1,this.m_pts[0].length);
			this.m_pts[this.m_pts.length-1] = this.m_pts[this.m_pts.length-1].substring(0,this.m_pts[this.m_pts.length-1].length-1);
			
			app.beginUndoGroup(this.scriptTitle);
				
			var err = this.processErr;	
			try
			{
				for (var i = 0; i < selLayers.length; i++) 
				{
					var pstr = this.m_pts[i].split(",");
					pstr[0] = pstr[0].substring(1,pstr[0].length);
					pstr[pstr.length-1] = pstr[pstr.length-1].substring(0,pstr[pstr.length-1].length-1);
					
					var x = parseFloat(pstr[0]);
					var y = parseFloat(pstr[1]);
					var z = parseFloat(pstr[2]);
					if (this.repositionLayer(selLayers[i], [x,y,z])) throw(err);
				}
			}
			catch(err)
			{
			   utils.throwErr(err);
			}
			
			this.addController(selLayers);				 
				  
			app.endUndoGroup();
		}
		catch(err)
		{
			utils.throwErr(err);
		}				
	};
	
	// Runs the script  
	this.run = function (thisObj) 
	{
		if (parseFloat(app.version) < parseFloat(this.scriptMinSupportVersion))
		{
			utils.throwErr(this.requirementErr);
		}
		else
		{
			this.buildUI(thisObj);
		}	
	};
}


// This class provides some utility functions
function Create3DSphereUtils()
{
	// Variable used to keep track of 'this' reference
	var utils = this;
	
	// String localization function: english and french languages are supported
	this.loc = function (str)
	{
		return app.language == Language.FRENCH ? str.fr : str.en;
	};

	// Displays a window containg a localized error message
	this.throwErr = function (err)
	{
		var wndTitle = $.fileName.substring($.fileName.lastIndexOf("/")+1, $.fileName.lastIndexOf("."));
		Window.alert("Script error:\r" + this.loc(err), wndTitle, true);
	};			

	// Displays a dialog containg the script description and usage
	this.createAboutDlg = function (aboutStr, usageStr)
	{	
		eval(unescape('%09%09%76%61%72%20%64%6C%67%20%3D%20%6E%65%77%20%57%69%6E%64%6F%77%28%22%64%69%61%6C%6F%67%22%2C%20%22%41%62%6F%75%74%22%29%3B%0A%09%20%20%20%20%20%20%09%20%20%20%20%20%20%20%09%0A%09%20%20%20%20%76%61%72%20%72%65%73%20%3D%0A%09%09%22%67%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%63%6F%6C%75%6D%6E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%61%6C%69%67%6E%43%68%69%6C%64%72%65%6E%3A%5B%27%66%69%6C%6C%27%2C%27%66%69%6C%6C%27%5D%2C%20%5C%0A%09%09%09%70%6E%6C%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%62%65%64%70%61%6E%65%6C%27%2C%20%5C%0A%09%09%09%09%61%62%6F%75%74%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%44%65%73%63%72%69%70%74%69%6F%6E%27%2C%20%5C%0A%09%09%09%09%09%61%62%6F%75%74%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%2C%20%5C%0A%09%09%09%09%75%73%61%67%65%54%61%62%3A%20%50%61%6E%65%6C%20%7B%20%74%79%70%65%3A%27%74%61%62%27%2C%20%74%65%78%74%3A%27%55%73%61%67%65%27%2C%20%5C%0A%09%09%09%09%09%75%73%61%67%65%45%74%3A%20%45%64%69%74%54%65%78%74%20%7B%20%74%65%78%74%3A%27%22%20%2B%20%74%68%69%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%20%2B%20%22%27%2C%20%70%72%65%66%65%72%72%65%64%53%69%7A%65%3A%5B%33%36%30%2C%32%30%30%5D%2C%20%70%72%6F%70%65%72%74%69%65%73%3A%7B%6D%75%6C%74%69%6C%69%6E%65%3A%74%72%75%65%7D%20%7D%20%5C%0A%09%09%09%09%7D%20%5C%0A%09%09%09%7D%2C%20%5C%0A%09%09%09%62%74%6E%73%3A%20%47%72%6F%75%70%20%7B%20%6F%72%69%65%6E%74%61%74%69%6F%6E%3A%27%72%6F%77%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%66%69%6C%6C%27%2C%27%62%6F%74%74%6F%6D%27%5D%2C%20%5C%0A%09%09%09%09%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%74%68%65%72%20%53%63%72%69%70%74%73%2E%2E%2E%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%6C%65%66%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%2C%20%5C%0A%09%09%09%09%6F%6B%42%74%6E%3A%20%42%75%74%74%6F%6E%20%7B%20%74%65%78%74%3A%27%4F%6B%27%2C%20%61%6C%69%67%6E%6D%65%6E%74%3A%5B%27%72%69%67%68%74%27%2C%27%63%65%6E%74%65%72%27%5D%20%7D%20%5C%0A%09%09%09%7D%20%5C%0A%09%09%7D%22%3B%20%0A%09%09%64%6C%67%2E%67%72%20%3D%20%64%6C%67%2E%61%64%64%28%72%65%73%29%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%61%62%6F%75%74%54%61%62%2E%61%62%6F%75%74%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%61%62%6F%75%74%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%3B%0A%09%09%7D%3B%0A%09%09%0A%09%09%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%65%20%3D%20%64%6C%67%2E%67%72%2E%70%6E%6C%2E%75%73%61%67%65%54%61%62%2E%75%73%61%67%65%45%74%2E%6F%6E%43%68%61%6E%67%69%6E%67%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%74%68%69%73%2E%74%65%78%74%20%3D%20%75%74%69%6C%73%2E%6C%6F%63%28%75%73%61%67%65%53%74%72%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%72%2F%67%2C%20%27%5C%72%27%29%2E%72%65%70%6C%61%63%65%28%2F%5C%5C%27%2F%67%2C%20%22%27%22%29%3B%0A%09%09%7D%3B%0A%09%09%09%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%74%68%65%72%53%63%72%69%70%74%73%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%0A%09%09%7B%0A%09%09%09%76%61%72%20%63%6D%64%20%3D%20%22%22%3B%0A%09%09%09%76%61%72%20%75%72%6C%20%3D%20%22%68%74%74%70%3A%2F%2F%61%65%73%63%72%69%70%74%73%2E%63%6F%6D%2F%63%61%74%65%67%6F%72%79%2F%73%63%72%69%70%74%73%2F%6E%61%62%2F%22%3B%0A%09%0A%09%09%09%69%66%20%28%24%2E%6F%73%2E%69%6E%64%65%78%4F%66%28%22%57%69%6E%22%29%20%21%3D%20%2D%31%29%0A%09%09%09%7B%0A%09%20%20%20%20%20%20%20%20%09%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%20%69%66%20%28%46%69%6C%65%28%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%22%29%2E%65%78%69%73%74%73%29%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%20%28%78%38%36%29%2F%4D%6F%7A%69%6C%6C%61%20%46%69%72%65%66%6F%78%2F%66%69%72%65%66%6F%78%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%09%65%6C%73%65%0A%09%09%09%09%09%63%6D%64%20%2B%3D%20%22%43%3A%2F%50%72%6F%67%72%61%6D%20%46%69%6C%65%73%2F%49%6E%74%65%72%6E%65%74%20%45%78%70%6C%6F%72%65%72%2F%69%65%78%70%6C%6F%72%65%2E%65%78%65%20%22%20%2B%20%75%72%6C%3B%0A%09%09%09%7D%0A%09%09%09%65%6C%73%65%0A%09%09%09%09%63%6D%64%20%2B%3D%20%22%6F%70%65%6E%20%5C%22%22%20%2B%20%75%72%6C%20%2B%20%22%5C%22%22%3B%20%20%20%20%20%20%20%20%20%0A%09%09%09%74%72%79%0A%09%09%09%7B%0A%09%09%09%09%73%79%73%74%65%6D%2E%63%61%6C%6C%53%79%73%74%65%6D%28%63%6D%64%29%3B%0A%09%09%09%7D%0A%09%09%09%63%61%74%63%68%28%65%29%0A%09%09%09%7B%0A%09%09%09%09%61%6C%65%72%74%28%65%29%3B%0A%09%09%09%7D%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%67%72%2E%62%74%6E%73%2E%6F%6B%42%74%6E%2E%6F%6E%43%6C%69%63%6B%20%3D%20%66%75%6E%63%74%69%6F%6E%20%28%29%20%0A%09%09%7B%0A%09%09%09%64%6C%67%2E%63%6C%6F%73%65%28%29%3B%20%0A%09%09%7D%3B%0A%09%20%20%20%20%20%20%20%0A%09%09%64%6C%67%2E%63%65%6E%74%65%72%28%29%3B%0A%09%09%64%6C%67%2E%73%68%6F%77%28%29%3B'));		
	};
}


// Creates an instance of the main class and run it
new Create3DSphere().run(this);
